import {
  View,
  Text,
  SafeAreaView,
  ScrollView,
  Button,
  Image,
  TouchableOpacity,
  Pressable,
  TextInput,
  Alert,
  Platform,
  Linking,
} from "react-native";
import React, { useEffect, useLayoutEffect, useRef, useState } from "react";

import { Ionicons } from "@expo/vector-icons";
import { COLORS } from "../styles/colors";
import { TextInputMask } from "react-native-masked-text";
import { KeyboardAwareScrollView } from "react-native-keyboard-aware-scroll-view";
import {
  addDoc,
  collection,
  doc,
  getDoc,
  getDocs,
  onSnapshot,
  query,
  updateDoc,
  where,
} from "firebase/firestore";
import { auth, db, storage } from "../../config/firebase";
import * as Haptics from "expo-haptics";
import moment from "moment";
import * as Location from "expo-location";
import * as ImagePicker from "expo-image-picker";
import {
  getDownloadURL,
  getStorage,
  ref,
  uploadBytes,
  uploadBytesResumable,
} from "firebase/storage";
import BackButton from "../components/Buttons/BackButton";
import {
  BUTTON_CLASS,
  BUTTON_OUTLINE_CLASS,
  INPUT_CLASS,
  INPUT_CLASS_BORDER_BOTTOM,
  INPUT_TEXT_AREA_CLASS,
} from "../styles/constants";
import { SENDGRID_API_KEY, SENDGRID_FROM } from "@env";
import axios from "axios";
import PrimaryButton from "../components/Buttons/PrimaryButton";
import { formatNumber } from "../utils/allFunctions";

export default function Findd({ navigation, route }) {
  const [licensePlate, setLicensePlate] = useState("");
  const [street, setStreet] = useState("");
  const [postalCode, setPostalCode] = useState("");
  const [city, setCity] = useState("");
  const [coords, setCoords] = useState();
  const [country, setCountry] = useState("");
  const [check, setCheck] = useState(false);
  const [image, setImage] = useState(null);
  const [isLoading, setIsLoading] = useState(false);

  useLayoutEffect(() => {
    navigation.setOptions({
      headerLeft: () => <BackButton onPress={navigation.goBack} />,
      headerTitle: "Déclarer un véhicule",
    });
  });

  async function getLocation() {
    let { status } = await Location.requestForegroundPermissionsAsync();
    if (status !== "granted") {
      Alert.alert(
        "Nous n'avons pas accès à votre position",
        "Veuillez nous l'accorder dans vos paramètres?",
        [
          {
            text: "Ouvrir les paramètres",
            onPress: () => Linking.openSettings(),
          },
          {
            text: "Annuler",
            style: "cancel",
          },
        ],
        {
          cancelable: true,
        }
      );

      return;
    }

    let geoReverse = await Location.getCurrentPositionAsync({
      accuracy: Location.Accuracy.Balanced,
    });
    let location = await Location.reverseGeocodeAsync(geoReverse.coords);
    setPostalCode(location[0].postalCode);
    setCity(location[0].city);
    setCountry(location[0].country);
    setStreet(location[0].name);
    const { longitude, latitude } = geoReverse.coords;
    setCoords({
      long: longitude,
      lat: latitude,
    });
    // setAddress(
    //   location[0].country +
    //     ", " +
    //     location[0].city +
    //     " (" +
    //     location[0].postalCode +
    //     ")" +
    //     ", " +
    //     location[0].street
    // );
  }

  async function checkForExistingSearch() {
    const q = query(
      collection(db, "searchs"),
      where("plate", "==", licensePlate),
      where("finderId", "==", ""),
      where("status", "==", "progress")
    );
    const querySnapshot = await getDocs(q);
    const dataArr = [];
    if (querySnapshot.size > 0) {
      querySnapshot.forEach((res) => {
        dataArr.push({ id: res.id, ...res.data() });
      });
      const findedSearchDemand = dataArr[0];
      const docSnap = await getDoc(doc(db, "users", findedSearchDemand.userId));
      if (docSnap.exists()) {
        sendPushNotifToSearcher({ id: docSnap.id, ...docSnap.data() });
        sendMailToSearcher({ id: docSnap.id, ...docSnap.data() });
        sendNotifToSearcher(docSnap.id);
      }

      await updateDoc(doc(db, "searchs", findedSearchDemand.id), {
        status: "ended",
        finderId: auth.currentUser.uid,
      });

      return true;
    } else {
      Alert.alert(
        "Votre annonce n'a pas été publiée pour une ou plusieurs raisons",
        "Soit un utilisateur a été plus rapide que vous,\nsoit aucune demande de recherche n'existe avec cette plaque d'immatriculation",
        [{ text: "Retour à l'accueil", onPress: () => navigation.popToTop() }]
      );
      return false;
    }
  }

  async function sendPushNotifToSearcher(searcherInfo) {
    const message = {
      to: searcherInfo.expoPushToken,
      sound: "default",
      title: "Vous avez un match ! ✅",
      body: "Votre signalement au sein de l'application Sur4plots a permis de retrouver votre véhicule",
    };

    await fetch("https://exp.host/--/api/v2/push/send", {
      method: "POST",
      headers: {
        Accept: "application/json",
        "Accept-encoding": "gzip, deflate",
        "Content-Type": "application/json",
      },
      body: JSON.stringify(message),
    });
  }

  function enleverEspaces(chaine) {
    return chaine.replace(/\s/g, "");
  }

  const sendMailToSearcher = async (searcherInfo) => {
    await axios.post(
      "https://api.sendgrid.com/v3/mail/send",
      {
        personalizations: [
          {
            to: [
              {
                email: `${searcherInfo.email}`,
              },
            ],
            dynamic_template_data: {
              first_name:
                searcherInfo.firstName !== ""
                  ? searcherInfo.firstName + " " + searcherInfo.lastName
                  : "cher utilisateur",
            },
          },
        ],
        from: {
          email: `${SENDGRID_FROM}`,
        },
        template_id: enleverEspaces("d-d8a99978650a4a6e9149b3cd7fd445de"),
      },
      {
        headers: {
          authorization: `Bearer ${SENDGRID_API_KEY}`,
          "Content-Type": "application/json",
        },
      }
    );
  };

  async function sendNotifToSearcher(searcherId) {
    const dataToAdd = {
      title: "Vous avez un match ! ✅",
      text:
        "Votre signalement au sein de l'application Sur4plots a permis de retrouver votre véhicule",
      userId: searcherId,
      isViewed: false,
      createdAt: moment().format(),
    };

    await addDoc(collection(db, "notifications"), dataToAdd);
  }

  async function validate() {
    setIsLoading(true);
    if (
      !licensePlate ||
      !street ||
      !city ||
      !country ||
      !postalCode ||
      !coords
    ) {
      Alert.alert("Certains champs sont vides", "Remplissez toutes les informations demandées");

      setIsLoading(false);
      return;
    }

    if (image == null) {
      Alert.alert("Ajoutez une photo","Montrez-nous une photo du véhicule retrouvé en question");

      setIsLoading(false);
      return;
    }
    const result = await checkForExistingSearch();
    const imgUrl = await uploadImage();

    if (!result) return;

    await addDoc(collection(db, "finds"), {
      address: {
        street,
        city,
        country,
        postalCode,
        coords,
      },
      searchPlate: licensePlate,
      userId: auth.currentUser.uid,
      imgUrl: imgUrl,
      status: "ended",
      paymentStatus: "unpaid",
      createdAt: moment().format(),
    }).then(() => {
      setCheck(true);
      setIsLoading(false);
    });
    setImage(null);
    setIsLoading(false);
  }

  const pickImage = async () => {
    const { status } = await ImagePicker.requestMediaLibraryPermissionsAsync();
    if (status !== "granted") {
      return;
    }

    const result = await ImagePicker.launchImageLibraryAsync({
      mediaTypes: ImagePicker.MediaTypeOptions.Images,
      allowsEditing: true,
      aspect: [4, 3],
      quality: 1,
    });

    if (!result.canceled) {
      setImage(result.assets[0].uri);
    }
  };

  const takePhotoWithCamera = async () => {
    const { status } = await ImagePicker.requestCameraPermissionsAsync();
    if (status !== "granted") {
      return;
    }
    const result = await ImagePicker.launchCameraAsync({
      allowsEditing: true,
      aspect: [4, 3],
      quality: 1,
    });
    if (!result.canceled) {
      setImage(result.assets[0].uri);
    }
  };

  const uploadImage = async () => {
    try {
      const response = await fetch(image);
      const blobFile = await response.blob();
      const storageRef = ref(storage, "finds/" + Date.now());

      const snapshot = await uploadBytesResumable(storageRef, blobFile);

      const imageUrl = await getDownloadURL(snapshot?.ref);
      return imageUrl;
    } catch (error) {
      // console.error("Error uploading or retrieving image:", error);
    }
  };

  if (check) {
    return (
      <View
        style={{ marginTop: "-50%" }}
        className="flex-1 items-center justify-center bg-white"
      >
      
        <View className="w-10/12 self-center">
          <Text
            className="mt-10 text-xl text-black text-center"
            style={{ fontFamily: "Inter_500Medium" }}
          >
            Votre signalement a bien été enregistré
          </Text>
          <Text className="mt-3 mb-5 text-base text-gray-700 text-center">
            {/* Vous recevrez une notification quand nous aurons des nouvelles */}
          </Text>

          <TouchableOpacity
            className={BUTTON_CLASS}
            onPress={navigation.popToTop}
          >
            <Text
              className="text-sm text-white"
              style={{ fontFamily: "Inter_500Medium" }}
            >
              Retour à l'accueil
            </Text>
          </TouchableOpacity>
        </View>
      </View>
    );
  }

  return (
    <SafeAreaView className="flex-1 bg-white">
      <KeyboardAwareScrollView
        keyboardDismissMode="interactive"
        keyboardShouldPersistTaps="handled"
        resetScrollToCoords={{ x: 0, y: 0 }}
        scrollEnabled
        contentContainerStyle={{ paddingBottom: "100%" }}
      >
        <View className="p-4 flex flex-row items-center">
          <View className="">
            <Text
              className="text-xl text-black"
              style={{ fontFamily: "Inter_500Medium" }}
            >
              Donnez-nous
            </Text>
            <Text
              className="text-base text-gray-500"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              plus d'informations
            </Text>
            <Text
              className="text-base text-gray-500"
              style={{ fontFamily: "Inter_400Regular" }}
            >
              sur le véhicule retrouvé
            </Text>
          </View>
        </View>
        <Pressable
          onPress={() =>
            Alert.alert(
              "Pourquoi avons nous besoin de la plaque d'immatriculation du véhicule ?",
              "Notre application permet de s'entraider dans le but de retrouver des véhicules volés/disparus, un utilisateur déclare avoir perdu un véhicule et un autre comme vous, déclare l'avoir retrouvé\nRentrer la plaque d'immatriculation nous permet de vérifier qu'il s'agit bel et bien du véhicule en question"
            )
          }
          className="rounded-xl px-4"
        >
          <View className="flex-row items-center justify-between">
            <View className="flex-row items-center">
              <Ionicons
                name="information-circle-outline"
                size={18}
                color={COLORS.primary}
              />
              <Text
                className="text-sm ml-2 underline"
                style={{ fontFamily: "Inter_300Light", color: COLORS.primary }}
              >
                Pourquoi avons nous besoin de la plaque d'immatriculation du
                véhicule ?
              </Text>
            </View>
          </View>
        </Pressable>

        <View className="px-5 mt-10">
          <View>
            <Text
              className="text-sm text-black mb-3"
              style={{ fontFamily: "Inter_500Medium" }}
            >
              Plaque d'immatriculation
            </Text>
            <TextInput
              className={INPUT_CLASS_BORDER_BOTTOM}
              placeholder="Sans espaces, sans caractères spéciaux"
              placeholderTextColor={"gray"}
              value={formatNumber(licensePlate, 3)}
              maxLength={15}
              onChangeText={(text) => setLicensePlate(text.toUpperCase())}
            />
            <Text
              className="text-xs text-gray-500 ml-1 mt-2"
              style={{ fontFamily: "Inter_300Light" }}
            >
              Exemple: Si votre plaque est: AA-123-BB, notez AA123BB
            </Text>
          </View>
          <View className="mt-8">
            <View className="mb-3 flex-row items-center justify-between">
              <Text
                className="text-sm text-black w-1/2"
                style={{ fontFamily: "Inter_500Medium" }}
              >
                Emplacement
              </Text>
              <TouchableOpacity
                style={{ borderColor: COLORS.primary }}
                className="py-1 px-2 rounded-md bg-gray-50 border"
                onPress={getLocation}
              >
                <Text
                  className="text-xs text-center"
                  style={{
                    fontFamily: "Inter_300Light",
                    color: COLORS.primary,
                  }}
                >
                  Remplir automatiquement
                </Text>
              </TouchableOpacity>
            </View>
            <TextInput
              className={INPUT_CLASS_BORDER_BOTTOM}
              placeholder="Rue / Boulevard"
              placeholderTextColor={"gray"}
              value={street}
              onChangeText={(val) => {
                setStreet(val);
              }}
            />
            <View className="mt-2 flex-row items-center justify-between">
              <TextInput
                style={{ width: "32%" }}
                className={INPUT_CLASS_BORDER_BOTTOM}
                placeholder={"Code postal"}
                placeholderTextColor={"gray"}
                value={postalCode}
                keyboardType="phone-pad"
                onChangeText={(val) => {
                  setPostalCode(val);
                }}
              />
              <TextInput
                style={{ width: "32%" }}
                className={INPUT_CLASS_BORDER_BOTTOM}
                placeholder="Ville"
                placeholderTextColor={"gray"}
                value={city}
                onChangeText={(val) => {
                  setCity(val);
                }}
              />
              <TextInput
                style={{ width: "32%" }}
                className={INPUT_CLASS_BORDER_BOTTOM}
                placeholder="Pays"
                placeholderTextColor={"gray"}
                value={country}
                onChangeText={(val) => {
                  setCountry(val);
                }}
              />
            </View>
            <Text
              className="text-xs text-gray-500 ml-1 mt-2"
              style={{ fontFamily: "Inter_300Light" }}
            >
              L'endroit ou vous l'avez vu pour la dernière fois, vous avez le
              choix de remplir automatiquement ou manuellement
            </Text>
          </View>

          <View className="mt-8">
            <View className="mb-3 flex-row items-center justify-between">
              <Text
                className="text-sm text-black w-1/2"
                style={{ fontFamily: "Inter_500Medium" }}
              >
                Photos
              </Text>
              <TouchableOpacity
                style={{ borderColor: COLORS.primary }}
                className="py-1 px-2 rounded-md bg-gray-50 border"
                onPress={takePhotoWithCamera}
              >
                <Text
                  className="text-xs  text-center"
                  style={{
                    fontFamily: "Inter_300Light",
                    color: COLORS.primary,
                  }}
                >
                  Utiliser la caméra
                </Text>
              </TouchableOpacity>
            </View>
            <Pressable
              className={
                "h-[18vh] my-2 bg-gray-100 border border-gray-200 w-full rounded-md flex-row items-center justify-center"
              }
              onPress={pickImage}
            >
              {image && (
                <Image
                  source={{ uri: image }}
                  className="w-full h-full rounded-xl"
                  resizeMode="cover"
                />
              )}
              <View
                style={{
                  position: image ? "absolute" : "relative",
                  borderColor: COLORS.primary,
                }}
                className="py-1 px-2 rounded-md bg-gray-50 border"
              >
                <Text
                  className={
                    !image ? "text-xs text-blue-500" : "text-sm text-blue-500"
                  }
                  style={{
                    fontFamily: "Inter_300Light",
                    color: COLORS.primary,
                  }}
                >
                  {!image ? "Importer une photo" : "Modifier la photo"}
                </Text>
              </View>
            </Pressable>
          </View>
          <View className="mt-5 w-full bg-white">
            <PrimaryButton
              text={"Publier"}
              isLoading={isLoading}
              disabled={isLoading}
              onPress={validate}
            />
          </View>
        </View>
      </KeyboardAwareScrollView>
    </SafeAreaView>
  );
}
